import { prisma } from './prisma';

export class SubscriptionError extends Error {
  constructor(
    message: string,
    public code: 'LIMIT_REACHED' | 'PLAN_REQUIRED' | 'FEATURE_UNAVAILABLE' | 'SUBSCRIPTION_INACTIVE'
  ) {
    super(message);
    this.name = 'SubscriptionError';
  }
}

export async function checkSubscriptionStatus(restaurantId: string) {
  const subscription = await prisma.subscription.findUnique({
    where: { restaurantId },
    include: { plan: true },
  });

  if (!subscription) {
    throw new SubscriptionError('Subscription not found', 'SUBSCRIPTION_INACTIVE');
  }

  const now = new Date();

  if (subscription.status === 'trialing') {
    if (subscription.trialEnd && subscription.trialEnd < now) {
      throw new SubscriptionError(
        'Trial-ul a expirat. Vă rugăm alegeți un plan pentru a continua.',
        'SUBSCRIPTION_INACTIVE'
      );
    }
    return subscription;
  }

  if (subscription.status === 'past_due') {
    if (subscription.gracePeriodEnd && subscription.gracePeriodEnd < now) {
      throw new SubscriptionError(
        'Abonamentul este suspendat din cauza plății eșuate.',
        'SUBSCRIPTION_INACTIVE'
      );
    }
    return subscription;
  }

  if (subscription.status !== 'active') {
    throw new SubscriptionError(
      'Abonamentul este inactiv. Vă rugăm contactați suportul.',
      'SUBSCRIPTION_INACTIVE'
    );
  }

  return subscription;
}

export async function enforceLimit(
  restaurantId: string,
  resource: 'products' | 'categories' | 'languages' | 'aiCredits'
) {
  const subscription = await checkSubscriptionStatus(restaurantId);
  const plan = subscription.plan;

  const counts = await prisma.restaurant.findUnique({
    where: { id: restaurantId },
    include: {
      _count: {
        select: {
          products: true,
          categories: true,
        },
      },
    },
  });

  if (!counts) {
    throw new Error('Restaurant not found');
  }

  switch (resource) {
    case 'products':
      if (counts._count.products >= plan.maxProducts) {
        throw new SubscriptionError(
          `Ați atins limita de ${plan.maxProducts} produse pentru planul ${plan.displayName}.`,
          'LIMIT_REACHED'
        );
      }
      break;

    case 'categories':
      if (counts._count.categories >= plan.maxCategories) {
        throw new SubscriptionError(
          `Ați atins limita de ${plan.maxCategories} categorii pentru planul ${plan.displayName}.`,
          'LIMIT_REACHED'
        );
      }
      break;

    case 'languages':
      break;

    case 'aiCredits':
      const usage = await prisma.usageCounter.findFirst({
        where: {
          subscriptionId: subscription.id,
          period: new Date(new Date().getFullYear(), new Date().getMonth(), 1),
        },
      });

      const creditsUsed = usage?.aiCreditsUsed || 0;

      if (creditsUsed >= plan.aiCreditsMonthly) {
        throw new SubscriptionError(
          `Ați atins limita de ${plan.aiCreditsMonthly} credite AI pentru luna aceasta.`,
          'LIMIT_REACHED'
        );
      }
      break;
  }
}

export async function checkFeatureAccess(
  restaurantId: string,
  feature: 'customDomain' | 'qrDownload' | 'multiLanguage'
) {
  const subscription = await checkSubscriptionStatus(restaurantId);
  const plan = subscription.plan;

  const featureMap = {
    customDomain: plan.allowCustomDomain,
    qrDownload: plan.allowQrDownload,
    multiLanguage: plan.maxLanguages > 1,
  };

  if (!featureMap[feature]) {
    throw new SubscriptionError(
      `Feature-ul "${feature}" nu este disponibil în planul ${plan.displayName}.`,
      'FEATURE_UNAVAILABLE'
    );
  }
}