import OpenAI from 'openai';

if (!process.env.OPENAI_API_KEY) {
  throw new Error('OPENAI_API_KEY is not set');
}

const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

export async function improveDescription(
  productName: string,
  currentDescription?: string
): Promise<{ description: string; tokensUsed: number }> {
  const prompt = currentDescription
    ? `Îmbunătățește următoarea descriere pentru produsul "${productName}" de restaurant. Păstrează limba română, maximum 50 cuvinte, fără emoji:\n\n${currentDescription}`
    : `Creează o descriere atrăgătoare pentru produsul de restaurant "${productName}". Limba română, maximum 50 cuvinte, fără emoji.`;

  const completion = await openai.chat.completions.create({
    model: 'gpt-4o-mini',
    messages: [
      {
        role: 'system',
        content: 'Ești un copywriter pentru restaurante. Creezi descrieri scurte, atrăgătoare, fără emoji.',
      },
      { role: 'user', content: prompt },
    ],
    max_tokens: 80,
    temperature: 0.7,
  });

  return {
    description: completion.choices[0].message.content?.trim() || '',
    tokensUsed: completion.usage?.total_tokens || 0,
  };
}

export async function translateToEnglish(
  nameRo: string,
  descriptionRo: string
): Promise<{ nameEn: string; descriptionEn: string; tokensUsed: number }> {
  const completion = await openai.chat.completions.create({
    model: 'gpt-4o-mini',
    messages: [
      {
        role: 'system',
        content: 'Translate Romanian restaurant menu items to English. Return JSON: { "nameEn": "...", "descriptionEn": "..." }',
      },
      { role: 'user', content: JSON.stringify({ nameRo, descriptionRo }) },
    ],
    max_tokens: 150,
    temperature: 0.3,
    response_format: { type: 'json_object' },
  });

  const result = JSON.parse(completion.choices[0].message.content || '{}');

  return {
    nameEn: result.nameEn || '',
    descriptionEn: result.descriptionEn || '',
    tokensUsed: completion.usage?.total_tokens || 0,
  };
}