import { NextRequest, NextResponse } from 'next/server';
import { headers } from 'next/headers';
import { stripe } from '@/lib/stripe';
import { prisma } from '@/lib/prisma';
import Stripe from 'stripe';
import { addDays } from 'date-fns';

const GRACE_PERIOD_DAYS = 3;

export async function POST(req: NextRequest) {
  const body = await req.text();
  const signature = headers().get('stripe-signature')!;

  let event: Stripe.Event;

  try {
    event = stripe.webhooks.constructEvent(
      body,
      signature,
      process.env.STRIPE_WEBHOOK_SECRET!
    );
  } catch (error: any) {
    console.error('Webhook signature verification failed:', error.message);
    return NextResponse.json({ error: 'Invalid signature' }, { status: 400 });
  }

  try {
    switch (event.type) {
      case 'checkout.session.completed': {
        const session = event.data.object as Stripe.Checkout.Session;
        const restaurantId = session.metadata?.restaurantId;
        const planName = session.metadata?.plan;
        const billingCycle = session.metadata?.billingCycle || 'monthly';

        if (!restaurantId || !planName) {
          throw new Error('Missing metadata in checkout session');
        }

        const plan = await prisma.plan.findUnique({
          where: { name: planName },
        });

        if (!plan) {
          throw new Error(`Plan not found: ${planName}`);
        }

        const stripeSubscription = await stripe.subscriptions.retrieve(
          session.subscription as string
        );

        await prisma.subscription.update({
          where: { restaurantId },
          data: {
            planId: plan.id,
            status: 'active',
            stripeCustomerId: session.customer as string,
            stripeSubscriptionId: session.subscription as string,
            stripePriceId: stripeSubscription.items.data[0].price.id,
            billingCycle: billingCycle as 'monthly' | 'yearly',
            currentPeriodStart: new Date(stripeSubscription.current_period_start * 1000),
            currentPeriodEnd: new Date(stripeSubscription.current_period_end * 1000),
            trialEnd: null,
            gracePeriodEnd: null,
            cancelAtPeriodEnd: false,
          },
        });

        console.log(`✅ Subscription activated: ${restaurantId} -> ${plan.displayName}`);
        break;
      }

      case 'customer.subscription.updated': {
        const subscription = event.data.object as Stripe.Subscription;

        const dbSubscription = await prisma.subscription.findUnique({
          where: { stripeSubscriptionId: subscription.id },
        });

        if (!dbSubscription) {
          console.warn(`Subscription not found in DB: ${subscription.id}`);
          break;
        }

        const updateData: any = {
          currentPeriodStart: new Date(subscription.current_period_start * 1000),
          currentPeriodEnd: new Date(subscription.current_period_end * 1000),
          cancelAtPeriodEnd: subscription.cancel_at_period_end,
        };

        if (subscription.status === 'active') {
          updateData.status = 'active';
          updateData.gracePeriodEnd = null;
        } else if (subscription.status === 'past_due') {
          updateData.status = 'past_due';
        } else if (subscription.status === 'canceled') {
          updateData.status = 'canceled';
          updateData.canceledAt = new Date();
        }

        if (subscription.items.data[0]?.price.id !== dbSubscription.stripePriceId) {
          const newPriceId = subscription.items.data[0].price.id;
          
          const newPlan = await prisma.plan.findFirst({
            where: {
              OR: [
                { stripePriceIdMonthly: newPriceId },
                { stripePriceIdYearly: newPriceId },
              ],
            },
          });

          if (newPlan) {
            updateData.planId = newPlan.id;
            updateData.stripePriceId = newPriceId;
            updateData.billingCycle = newPlan.stripePriceIdYearly === newPriceId ? 'yearly' : 'monthly';
            console.log(`Plan changed: ${dbSubscription.restaurantId} -> ${newPlan.displayName}`);
          }
        }

        await prisma.subscription.update({
          where: { id: dbSubscription.id },
          data: updateData,
        });

        console.log(`✅ Subscription updated: ${subscription.id}`);
        break;
      }

      case 'customer.subscription.deleted': {
        const subscription = event.data.object as Stripe.Subscription;

        const dbSubscription = await prisma.subscription.findUnique({
          where: { stripeSubscriptionId: subscription.id },
        });

        if (!dbSubscription) {
          console.warn(`Subscription not found in DB: ${subscription.id}`);
          break;
        }

        const trialPlan = await prisma.plan.findUnique({
          where: { name: 'trial' },
        });

        if (trialPlan) {
          await prisma.subscription.update({
            where: { id: dbSubscription.id },
            data: {
              planId: trialPlan.id,
              status: 'canceled',
              canceledAt: new Date(),
              stripeSubscriptionId: null,
              stripePriceId: null,
            },
          });
        }

        console.log(`✅ Subscription canceled: ${subscription.id}`);
        break;
      }

      case 'invoice.payment_failed': {
        const invoice = event.data.object as Stripe.Invoice;

        const subscription = await prisma.subscription.findUnique({
          where: { stripeCustomerId: invoice.customer as string },
        });

        if (!subscription) {
          console.warn(`Subscription not found for customer: ${invoice.customer}`);
          break;
        }

        const gracePeriodEnd = addDays(new Date(), GRACE_PERIOD_DAYS);

        await prisma.subscription.update({
          where: { id: subscription.id },
          data: {
            status: 'past_due',
            gracePeriodEnd,
          },
        });

        console.log(`⚠️ Payment failed, grace period until: ${gracePeriodEnd.toISOString()}`);
        break;
      }

      case 'invoice.payment_succeeded': {
        const invoice = event.data.object as Stripe.Invoice;

        const subscription = await prisma.subscription.findUnique({
          where: { stripeCustomerId: invoice.customer as string },
        });

        if (!subscription) {
          break;
        }

        if (subscription.status === 'past_due') {
          await prisma.subscription.update({
            where: { id: subscription.id },
            data: {
              status: 'active',
              gracePeriodEnd: null,
            },
          });

          console.log(`✅ Payment recovered for: ${subscription.restaurantId}`);
        }
        break;
      }

      default:
        console.log(`Unhandled event type: ${event.type}`);
    }

    return NextResponse.json({ received: true });
  } catch (error: any) {
    console.error('Webhook handler error:', error);
    return NextResponse.json({ error: error.message }, { status: 500 });
  }
}