import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { productSchema } from '@/lib/validations';
import { enforceLimit, SubscriptionError } from '@/lib/subscription-guard';
import { incrementUsage } from '@/lib/usage';

export async function GET(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session?.user) {
      return NextResponse.json({ error: 'Neautentificat' }, { status: 401 });
    }

    const restaurant = await prisma.restaurant.findFirst({
      where: { ownerId: session.user.id },
    });

    if (!restaurant) {
      return NextResponse.json({ error: 'Restaurant negăsit' }, { status: 404 });
    }

    const { searchParams } = new URL(req.url);
    const categoryId = searchParams.get('categoryId');

    const products = await prisma.product.findMany({
      where: {
        restaurantId: restaurant.id,
        ...(categoryId && { categoryId }),
      },
      orderBy: { sortOrder: 'asc' },
      include: {
        category: {
          select: {
            nameRo: true,
            nameEn: true,
          },
        },
      },
    });

    return NextResponse.json(products);
  } catch (error) {
    console.error('Get products error:', error);
    return NextResponse.json({ error: 'Eroare server' }, { status: 500 });
  }
}

export async function POST(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session?.user) {
      return NextResponse.json({ error: 'Neautentificat' }, { status: 401 });
    }

    const restaurant = await prisma.restaurant.findFirst({
      where: { ownerId: session.user.id },
      include: { subscription: true },
    });

    if (!restaurant) {
      return NextResponse.json({ error: 'Restaurant negăsit' }, { status: 404 });
    }

    await enforceLimit(restaurant.id, 'products');

    const body = await req.json();
    const data = productSchema.parse(body);

    const maxSortOrder = await prisma.product.findFirst({
      where: {
        restaurantId: restaurant.id,
        categoryId: data.categoryId,
      },
      orderBy: { sortOrder: 'desc' },
      select: { sortOrder: true },
    });

    const product = await prisma.product.create({
      data: {
        restaurantId: restaurant.id,
        ...data,
        sortOrder: (maxSortOrder?.sortOrder || 0) + 1,
      },
    });

    if (restaurant.subscription) {
      await incrementUsage(restaurant.subscription.id, 'products', 1);
    }

    await prisma.auditLog.create({
      data: {
        userId: session.user.id,
        action: 'CREATE',
        entity: 'product',
        entityId: product.id,
        metadata: { name: product.nameRo },
      },
    });

    return NextResponse.json(product);
  } catch (error: any) {
    if (error instanceof SubscriptionError) {
      return NextResponse.json(
        { error: error.message, code: error.code },
        { status: 403 }
      );
    }
    console.error('Create product error:', error);
    return NextResponse.json({ error: 'Eroare server' }, { status: 500 });
  }
}