import { NextRequest, NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { translateToEnglish } from '@/lib/openai';
import { enforceLimit, SubscriptionError } from '@/lib/subscription-guard';
import { incrementUsage } from '@/lib/usage';

export async function POST(req: NextRequest) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session?.user) {
      return NextResponse.json({ error: 'Neautentificat' }, { status: 401 });
    }

    const { productId } = await req.json();

    const product = await prisma.product.findUnique({
      where: { id: productId },
      include: {
        restaurant: {
          include: { subscription: { include: { plan: true } } },
        },
      },
    });

    if (!product || product.restaurant.ownerId !== session.user.id) {
      return NextResponse.json({ error: 'Produs negăsit' }, { status: 404 });
    }

    await enforceLimit(product.restaurantId, 'languages');
    await enforceLimit(product.restaurantId, 'aiCredits');

    const { nameEn, descriptionEn, tokensUsed } = await translateToEnglish(
      product.nameRo,
      product.descriptionRo || ''
    );

    await prisma.product.update({
      where: { id: productId },
      data: { nameEn, descriptionEn },
    });

    const costPer1kTokens = 0.00015;
    const costUsd = (tokensUsed / 1000) * costPer1kTokens;

    await prisma.aiUsage.create({
      data: {
        restaurantId: product.restaurantId,
        type: 'translate',
        productId,
        tokensUsed,
        costUsd,
      },
    });

    if (product.restaurant.subscription) {
      const creditsUsed = product.restaurant.subscription.plan.aiCreditsPerCall;
      await incrementUsage(product.restaurant.subscription.id, 'aiCredits', creditsUsed);
    }

    return NextResponse.json({ nameEn, descriptionEn, tokensUsed });
  } catch (error: any) {
    if (error instanceof SubscriptionError) {
      return NextResponse.json(
        { error: error.message, code: error.code },
        { status: 403 }
      );
    }
    console.error('AI translate error:', error);
    return NextResponse.json({ error: 'Eroare la traducere' }, { status: 500 });
  }
}